<?php
declare(strict_types=1);

namespace App\Services;

use App\Core\DB;

final class GeoService
{
  public function geocode(string $addressText): ?array
  {
    $q = trim($addressText);
    if ($q === '') return null;

    $hash = hash('sha256', mb_strtolower($q));

    $pdo = DB::pdo();
    $st = $pdo->prepare("SELECT lat,lng FROM geocode_cache WHERE query_hash=? LIMIT 1");
    $st->execute([$hash]);
    $row = $st->fetch();
    if ($row) return ['lat' => (float)$row['lat'], 'lng' => (float)$row['lng'], 'source' => 'db_cache'];

    $cached = Cache::get('geocode:' . $hash);
    if (Cache::valid($cached) && isset($cached['lat'], $cached['lng'])) {
      return ['lat' => (float)$cached['lat'], 'lng' => (float)$cached['lng'], 'source' => 'file_cache'];
    }

    $url = $_ENV['NOMINATIM_URL'] ?? 'https://nominatim.openstreetmap.org/search';
    $ua  = $_ENV['NOMINATIM_USER_AGENT'] ?? 'GuiaLocalMVP/1.0';

    $params = http_build_query(['q' => $q, 'format' => 'json', 'limit' => 1]);

    $ch = curl_init($url . '?' . $params);
    curl_setopt_array($ch, [
      CURLOPT_RETURNTRANSFER => true,
      CURLOPT_TIMEOUT => 8,
      CURLOPT_HTTPHEADER => [
        'User-Agent: ' . $ua,
        'Accept: application/json'
      ],
    ]);

    $resp = curl_exec($ch);
    $code = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    if ($resp === false || $code < 200 || $code >= 300) return null;

    $json = json_decode($resp, true);
    if (!is_array($json) || empty($json[0]['lat']) || empty($json[0]['lon'])) return null;

    $lat = (float)$json[0]['lat'];
    $lng = (float)$json[0]['lon'];

    Cache::set('geocode:' . $hash, ['lat'=>$lat, 'lng'=>$lng], 60*60*24*30);

    $ins = $pdo->prepare("INSERT INTO geocode_cache (query_hash,query_text,lat,lng,raw_json) VALUES (?,?,?,?,?)");
    $ins->execute([$hash, mb_substr($q, 0, 255), $lat, $lng, json_encode($json[0])]);

    return ['lat' => $lat, 'lng' => $lng, 'source' => 'nominatim'];
  }
}
